//-----------------------------------------------------------------------
// <copyright file="HelloARController.cs" company="Google LLC">
//
// Copyright 2020 Google LLC. All Rights Reserved.
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
// http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.
//
// </copyright>
//-----------------------------------------------------------------------

namespace GoogleARCore.Examples.HelloAR
{
    using System.Collections.Generic;
    using System.Threading;
    using GoogleARCore;
    using GoogleARCore.Examples.Common;
    using UnityEngine;
    using UnityEngine.EventSystems;
    using UnityEngine.UI;

#if UNITY_EDITOR
    // Set up touch input propagation while using Instant Preview in the editor.
    using Input = InstantPreviewInput;
#endif

    /// <summary>
    /// Controls the HelloAR example.
    /// </summary>
    public class HelloARController : MonoBehaviour
    {
        /// <summary>
        /// The Depth Setting Menu.
        /// </summary>
        public DepthMenu DepthMenu;

        /// <summary>
        /// The first-person camera being used to render the passthrough camera image (i.e. AR
        /// background).
        /// </summary>
        public Camera FirstPersonCamera;

        /// <summary>
        /// A prefab to place when a raycast from a user touch hits a plane.
        /// </summary>
        public GameObject pointPrefab;

        /// <summary>
        /// A prefab to place to unite two adiacent points.
        /// </summary>
        public GameObject linePrefab;

        /// <summary>
        /// A prefab to place to display the distance between two adiacent points.
        /// </summary>
        public Text textPrefab;

        /// <summary>
        /// The canvas needed to display text on screen.
        /// </summary>
        public Canvas parent;

        /// <summary>
        /// A list of all added points on screen.
        /// </summary>
        public List<GameObject> points = new List<GameObject>();

        /// <summary>
        /// A list of distances of adiacent points on screen.
        /// </summary>
        public List<Text> distances = new List<Text>();

        /// <summary>
        /// The total distance of all points on screen.
        /// </summary>
        public Text totalDistance;
        float distanceSum = 0;

        /// <summary>
        /// A prefab to place when a raycast from a user touch hits a feature point.
        /// </summary>
        public GameObject GameObjectPointPrefab;

        /// <summary>
        /// The rotation in degrees need to apply to prefab when it is placed.
        /// </summary>
        private const float k_PrefabRotation = 180.0f;

        /// <summary>
        /// True if the app is in the process of quitting due to an ARCore connection error,
        /// otherwise false.
        /// </summary>
        private bool m_IsQuitting = false;

        /// <summary>
        /// The Unity Awake() method.
        /// </summary>
        public void Awake()
        {
            // Enable ARCore to target 60fps camera capture frame rate on supported devices.
            // Note, Application.targetFrameRate is ignored when QualitySettings.vSyncCount != 0.
            Application.targetFrameRate = 60;
        }

        /// <summary>
        /// The Unity Update() method.
        /// </summary>
        public void Update()
        {
            _UpdateApplicationLifecycle();

            if (DepthMenu != null && !DepthMenu.CanPlaceAsset())
            {
                return;
            }

            // The distance between two points needs to be always placed near the two points.
            // If we move the phone screen the text will also move (the text is fixed on canvas
            // ant the canvas always follows the phone screen),
            // so we need to update the position on screen for each distance displayed
            for (int i = 0; i < distances.Count; i++)
            {
                // TODO 2.3 Update text position - SOLVE THIS AFTER TESTING 2.1 - 2.2 AND NOTICE THE DIFFERENCES
                distances[i].transform.position = new Vector3(0, 0, 0);
            }

            // If the player has not touched the screen, we are done with this update.
            Touch touch;
            if (Input.touchCount < 1 || (touch = Input.GetTouch(0)).phase != TouchPhase.Began)
            {
                return;
            }

            // Should not handle input if the player is pointing on UI.
            if (EventSystem.current.IsPointerOverGameObject(touch.fingerId))
            {
                return;
            }

            // Raycast against the location the player touched to search for planes.
            TrackableHit hit;
            TrackableHitFlags raycastFilter = TrackableHitFlags.PlaneWithinPolygon |
                TrackableHitFlags.FeaturePointWithSurfaceNormal;

            if (Frame.Raycast(touch.position.x, touch.position.y, raycastFilter, out hit))
            {
                // Use hit pose and camera pose to check if hittest is from the
                // back of the plane, if it is, no need to create the anchor.
                if ((hit.Trackable is DetectedPlane) &&
                    Vector3.Dot(FirstPersonCamera.transform.position - hit.Pose.position,
                        hit.Pose.rotation * Vector3.up) < 0)
                {
                    Debug.Log("Hit at back of the current DetectedPlane");
                }
                else
                {
                    if (DepthMenu != null)
                    {
                        // Show depth card window if necessary.
                        DepthMenu.ConfigureDepthBeforePlacingFirstAsset();
                    }

                    // Instantiate pointPrefab at the hit pose.
                    var gameObject = Instantiate(pointPrefab, hit.Pose.position, hit.Pose.rotation);

                    // Compensate for the hitPose rotation facing away from the raycast (i.e.
                    // camera).
                    gameObject.transform.Rotate(0, k_PrefabRotation, 0, Space.Self);

                    // Create an anchor to allow ARCore to track the hitpoint as understanding of
                    // the physical world evolves.
                    var anchor = hit.Trackable.CreateAnchor(hit.Pose);

                    // Make game object a child of the anchor.
                    gameObject.transform.parent = anchor.transform;

                    // Add a point to the list of points
                    points.Add(gameObject);
                    
                    // If there is more than one point on screen, we can compute the distance
                    if (points.Count > 1)
                    {
                        // Draw a line between the last added two points
                        // - TODO 1.1 Instantiate linePrefab
                        GameObject line = new GameObject();
                        // - TODO 1.2 Set position at half the distance between the last two added points
                        line.transform.position = new Vector3(points[points.Count - 1].transform.position.x 
                            + (points[points.Count - 2].transform.position.x 
                            - points[points.Count - 1].transform.position.x) / 2, 
                            0, 0);
                        // - Set rotation
                        line.transform.LookAt(points[points.Count - 1].transform);
                        // - TODO 1.3 Set scale: two fixed numbers on ox and oy axis, the distance between the two points on oz axis 
                        line.transform.localScale = new Vector3(0.005f, 0.005f, 0); // !0.005 can be changed to whatever value we want
                        // - TODO 1.4 Set anchor so that the line will not move - SOLVE THIS AFTER TESTING 1.1 - 1.3 AND NOTICE THE DIFFERENCES
                        

                        // Show on each line the distance
                        // - Instantiate textPrefab
                        Text partialDistance = Instantiate(textPrefab);
                        // - Set canvas as parent so that the text is displayed on screen
                        partialDistance.transform.SetParent(parent.transform);
                        // - Set position on screen
                        partialDistance.transform.position = Camera.main.WorldToScreenPoint(new Vector3(line.transform.position.x, 
                            line.transform.position.y, line.transform.position.z));
                        // - TODO 2.1 Compute distance between the two last added points
                        float deltaX = 0;
                        float deltaY = 0;
                        float deltaZ = 0;
                        float distance = (float)Mathf.Sqrt(deltaX * deltaX + deltaY * deltaY + deltaZ * deltaZ);
                        // - TODO 2.2 Add the distance to our distances list
                        partialDistance.text = "";
                        distances.Add(partialDistance);

                        // TODO 3.1 Update total distance
                        distanceSum = 0;
                        totalDistance.text = "Total distance: " + distanceSum.ToString();
                    }
                }
            }
        }

        /// <summary>
        /// Check and update the application lifecycle.
        /// </summary>
        private void _UpdateApplicationLifecycle()
        {
            // Exit the app when the 'back' button is pressed.
            if (Input.GetKey(KeyCode.Escape))
            {
                Application.Quit();
            }

            // Only allow the screen to sleep when not tracking.
            if (Session.Status != SessionStatus.Tracking)
            {
                Screen.sleepTimeout = SleepTimeout.SystemSetting;
            }
            else
            {
                Screen.sleepTimeout = SleepTimeout.NeverSleep;
            }

            if (m_IsQuitting)
            {
                return;
            }

            // Quit if ARCore was unable to connect and give Unity some time for the toast to
            // appear.
            if (Session.Status == SessionStatus.ErrorPermissionNotGranted)
            {
                _ShowAndroidToastMessage("Camera permission is needed to run this application.");
                m_IsQuitting = true;
                Invoke("_DoQuit", 0.5f);
            }
            else if (Session.Status.IsError())
            {
                _ShowAndroidToastMessage(
                    "ARCore encountered a problem connecting.  Please start the app again.");
                m_IsQuitting = true;
                Invoke("_DoQuit", 0.5f);
            }
        }

        /// <summary>
        /// Actually quit the application.
        /// </summary>
        private void _DoQuit()
        {
            Application.Quit();
        }

        /// <summary>
        /// Show an Android toast message.
        /// </summary>
        /// <param name="message">Message string to show in the toast.</param>
        private void _ShowAndroidToastMessage(string message)
        {
            AndroidJavaClass unityPlayer = new AndroidJavaClass("com.unity3d.player.UnityPlayer");
            AndroidJavaObject unityActivity =
                unityPlayer.GetStatic<AndroidJavaObject>("currentActivity");

            if (unityActivity != null)
            {
                AndroidJavaClass toastClass = new AndroidJavaClass("android.widget.Toast");
                unityActivity.Call("runOnUiThread", new AndroidJavaRunnable(() =>
                {
                    AndroidJavaObject toastObject =
                        toastClass.CallStatic<AndroidJavaObject>(
                            "makeText", unityActivity, message, 0);
                    toastObject.Call("show");
                }));
            }
        }
    }
}
